<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Models\Content;
use App\Models\Contoh;
use App\Models\Category;

class MenuController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $menus = Menu::all();
        return view('menu.index', compact('menus'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('menu.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'icon' => 'nullable|file|image',
            'description' => 'nullable|string|max:255',
            'order' => 'nullable|integer',
            'status' => 'nullable|boolean',
        ]);
        if ($request->hasFile('icon')) {
            $file = $request->file('icon');
            $path = $file->store('menu_icons', 'public');
            $validated['icon'] = $path;
        } else {
            $validated['icon'] = null;
        }
        Menu::create($validated);
        return redirect()->route('home')->with('success', 'Menu berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Menu $menu)
    {
        $menu->load(['contents.category', 'contents.refleksiUser', 'contents.feedback']);
        // Ambil kategori unik dari semua content pada menu ini
        $categories = $menu->contents
            ->whereNotNull('category_id')
            ->map(function($content) { return $content->category; })
            ->filter()
            ->unique('id')
            ->values();
        $allCategories = \App\Models\Category::all();
        return view('menu.show', compact('menu', 'categories', 'allCategories'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Menu $menu)
    {
        return view('menu.edit', compact('menu'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Menu $menu)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'icon' => 'nullable|file|image',
            'description' => 'nullable|string|max:255',
            'order' => 'nullable|integer',
            'status' => 'nullable|boolean',
        ]);
        if ($request->hasFile('icon')) {
            $file = $request->file('icon');
            $path = $file->store('menu_icons', 'public');
            $validated['icon'] = $path;
        } else {
            unset($validated['icon']); // jangan update kolom icon jika tidak upload baru
        }
        $menu->update($validated);
        return redirect()->route('home')->with('success', 'Menu berhasil diupdate.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Menu $menu)
    {
        $menu->delete();
        return redirect()->route('home')->with('success', 'Menu berhasil dihapus.');
    }

    /**
     * API: Ambil semua data menu (JSON)
     */
    public function apiIndex()
    {
        $host = 'https://tisel.pthas.or.id';
        $menus = Menu::where('status', 1)->get()->map(function($menu) use ($host) {
            $menuArr = $menu->toArray();
            if ($menu->icon) {
                $iconPath = 'storage/' . ltrim(str_replace('storage/', '', $menu->icon), '/');
                $menuArr['icon'] = $host . '/' . ltrim($iconPath, '/');
            }
            return $menuArr;
        });
        return response()->json(['data' => $menus], 200, [], JSON_UNESCAPED_SLASHES);
    }

    /**
     * Tampilkan konten berdasarkan menu
     */
    public function showContents(Menu $menu)
    {
        $contents = $menu->contents()->where('status', 'published')->latest()->paginate(10);
        return view('menu.contents', compact('menu', 'contents'));
    }

    /**
     * API: Detail menu beserta kategori & konten per kategori
     */
    public function apiMenuDetail($menu)
    {
        // Cari menu manual
        $menu = \App\Models\Menu::find($menu);
        // Debug log
        \Log::info('API Menu Detail Debug', ['menu' => $menu]);
        // Cek validitas menu
        if (!$menu || !$menu->id) {
            return response()->json(['error' => 'Menu not found'], 404);
        }
        $menu->load('contents.category');
        // Ambil kategori unik dari semua content pada menu ini
        $categories = $menu->contents
            ->whereNotNull('category_id')
            ->map(function($content) { return $content->category; })
            ->filter()
            ->unique('id')
            ->values();

        $categoriesArr = $categories->map(function($category) use ($menu) {
            return [
                'id' => $category->id,
                'name' => $category->name,
                'description' => $category->description,
                'contents' => $menu->contents
                    ->where('category_id', $category->id)
                    ->values()
                    ->map(function($content) {
                        return [
                            'id' => $content->id,
                            'title' => $content->title,
                            'description' => $content->description,
                            'content' => $content->content,
                            'status' => $content->status,
                        ];
                    }),
            ];
        });

        return response()->json([
            'menu' => [
                'id' => $menu->id,
                'name' => $menu->name,
                'description' => $menu->description,
                'icon' => $menu->icon,
                'status' => $menu->status,
            ],
            'categories' => $categoriesArr,
        ]);
    }

    /**
     * Import data menus, contents, contoh dari file Excel/CSV
     */
    public function import(Request $request)
    {
        // Hanya admin/superadmin
        if (!auth()->check() || !auth()->user()->hasAnyRole(['admin', 'superadmin'])) {
            abort(403, 'Unauthorized');
        }
        $request->validate([
            'import_file' => 'required|file|mimes:xlsx,csv,txt',
        ]);

        $file = $request->file('import_file');
        $path = $file->getRealPath();
        $spreadsheet = IOFactory::load($path);
        $success = 0;
        $fail = 0;
        // Sheet: menus
        $menuSheet = $spreadsheet->getSheetByName('menus');
        if ($menuSheet) {
            $rows = $menuSheet->toArray(null, true, true, true);
            $header = array_map('strtolower', $rows[1]);
            for ($i = 2; $i <= count($rows); $i++) {
                $row = $rows[$i];
                if (empty($row['A'])) continue;
                $data = array_combine($header, $row);
                try {
                    \App\Models\Menu::updateOrCreate(
                        ['name' => $data['name']],
                        [
                            'description' => $data['description'] ?? null,
                            'order' => $data['order'] ?? null,
                            'status' => $data['status'] ?? 1,
                        ]
                    );
                    $success++;
                } catch (\Exception $e) {
                    $fail++;
                }
            }
        }
        // Sheet: contents
        $contentSheet = $spreadsheet->getSheetByName('contents');
        if ($contentSheet) {
            $rows = $contentSheet->toArray(null, true, true, true);
            $header = array_map('strtolower', $rows[1]);
            for ($i = 2; $i <= count($rows); $i++) {
                $row = $rows[$i];
                if (empty($row['A'])) continue;
                $data = array_combine($header, $row);
                $menu = \App\Models\Menu::where('name', $data['menu_name'])->first();
                if (!$menu) continue;
                try {
                    Content::updateOrCreate(
                        ['title' => $data['title'], 'menu_id' => $menu->id],
                        [
                            'category_id' => $data['category_id'] ?? null,
                            'content' => $data['content'] ?? null,
                            'status' => $data['status'] ?? 'published',
                            'menu_id' => $menu->id,
                        ]
                    );
                    $success++;
                } catch (\Exception $e) {
                    $fail++;
                }
            }
        }
        // Sheet: contohhs
        $contohSheet = $spreadsheet->getSheetByName('contohs');
        if ($contohSheet) {
            $rows = $contohSheet->toArray(null, true, true, true);
            $header = array_map('strtolower', $rows[1]);
            for ($i = 2; $i <= count($rows); $i++) {
                $row = $rows[$i];
                if (empty($row['A'])) continue;
                $data = array_combine($header, $row);
                $content = Content::where('title', $data['content_title'])->first();
                if (!$content) continue;
                try {
                    Contoh::updateOrCreate(
                        ['nama' => $data['nama'], 'content_id' => $content->id],
                        [
                            'isi_contoh' => $data['isi_contoh'] ?? null,
                            'content_id' => $content->id,
                        ]
                    );
                    $success++;
                } catch (\Exception $e) {
                    $fail++;
                }
            }
        }
        return redirect()->back()->with('success', "Import selesai. Berhasil: $success, Gagal: $fail");
    }

    public function downloadTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        // Sheet 1: menus
        $sheetMenus = $spreadsheet->getActiveSheet();
        $sheetMenus->setTitle('menus');
        $sheetMenus->fromArray([
            ['name', 'description', 'order', 'status'],
            ['Menu 1', 'Deskripsi menu 1', 1, 1],
        ]);
        // Sheet 2: contents
        $sheetContents = new \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet($spreadsheet, 'contents');
        $sheetContents->fromArray([
            ['menu_name', 'category_id', 'title', 'content', 'status'],
            ['Menu 1', 1, 'Judul Konten', 'Isi konten html', 'published'],
        ]);
        $spreadsheet->addSheet($sheetContents);
        // Sheet 3: contohhs
        $sheetContoh = new \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet($spreadsheet, 'contohs');
        $sheetContoh->fromArray([
            ['content_title', 'nama', 'isi_contoh'],
            ['Judul Konten', 'Contoh 1', 'Isi contoh 1'],
        ]);
        $spreadsheet->addSheet($sheetContoh);
        $spreadsheet->setActiveSheetIndex(0);
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $filename = 'import_template.xlsx';
        return response()->streamDownload(function() use ($writer) {
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }
}
